<?php

namespace App\Http\Controllers\Customer;

use App\Http\Controllers\Controller;
use App\Models\Reservation;
use App\Models\Outlet;
use App\Models\Table;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

/**
 * Customer-facing Reservation Controller
 *
 * Handles table reservations from customer perspective
 */
class ReservationController extends Controller
{
    /**
     * Show reservation booking form
     */
    public function create()
    {
        $outlets = Outlet::all();
        $tables = Table::all();

        // Get available time slots (example: 10:00 - 22:00, every 30 minutes)
        $timeSlots = $this->generateTimeSlots('10:00', '22:00', 30);

        return view('customer.reservations.create', compact('outlets', 'tables', 'timeSlots'));
    }

    /**
     * Store new reservation from customer
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'outlet_id' => 'required|exists:outlets,id',
            'reservation_date' => 'required|date|after_or_equal:today',
            'reservation_time' => 'required|date_format:H:i',
            'party_size' => 'required|integer|min:1|max:20',
            'special_requests' => 'nullable|string|max:500',
        ]);

        // If user is not logged in, validate guest info
        if (!Auth::check()) {
            $request->validate([
                'guest_name' => 'required|string|max:255',
                'guest_email' => 'required|email|max:255',
                'guest_phone' => 'required|string|max:20',
            ]);
        }

        // Generate unique reservation number
        $reservationNumber = 'RSV-' . Str::upper(Str::random(10));
        while (Reservation::where('reservation_number', $reservationNumber)->exists()) {
            $reservationNumber = 'RSV-' . Str::upper(Str::random(10));
        }

        // Check if time slot still available (basic check)
        $conflictingReservation = Reservation::where('outlet_id', $validated['outlet_id'])
            ->where('reservation_date', $validated['reservation_date'])
            ->where('reservation_time', $validated['reservation_time'])
            ->whereIn('status', ['pending', 'confirmed'])
            ->count();

        // Allow max 3 reservations per time slot (configurable)
        if ($conflictingReservation >= 3) {
            return back()->with('error', 'Maaf, waktu yang dipilih sudah penuh. Silakan pilih waktu lain.')
                ->withInput();
        }

        // Create reservation
        $reservationData = [
            'reservation_number' => $reservationNumber,
            'user_id' => Auth::id(),
            'outlet_id' => $validated['outlet_id'],
            'table_id' => null, // Will be assigned by cashier later
            'reservation_date' => $validated['reservation_date'],
            'reservation_time' => $validated['reservation_time'],
            'party_size' => $validated['party_size'],
            'status' => Reservation::STATUS_PENDING, // Pending approval
            'special_requests' => $validated['special_requests'],
        ];

        // Add guest info if not logged in
        if (!Auth::check()) {
            $reservationData['guest_info'] = [
                'name' => $request->guest_name,
                'email' => $request->guest_email,
                'phone' => $request->guest_phone,
            ];
        }

        $reservation = Reservation::create($reservationData);

        // Store in session for guest tracking
        if (!Auth::check()) {
            $guestReservations = Session::get('guest_reservations', []);
            $guestReservations[] = $reservation->id;
            Session::put('guest_reservations', array_unique($guestReservations));
        }

        Log::info('Customer reservation created', [
            'reservation_number' => $reservation->reservation_number,
            'user_id' => Auth::id() ?? 'guest',
            'customer' => Auth::check() ? Auth::user()->name : $request->guest_name,
        ]);

        // TODO: Send confirmation email to customer

        return redirect()->route('customer.reservations.show', $reservation->reservation_number)
            ->with('success', 'Reservasi berhasil dibuat! Nomor reservasi Anda: ' . $reservation->reservation_number);
    }

    /**
     * Show customer's reservations
     */
    public function index()
    {
        if (Auth::check()) {
            // For logged in users
            $reservations = Reservation::where('user_id', Auth::id())
                ->orderBy('reservation_date', 'desc')
                ->orderBy('reservation_time', 'desc')
                ->with(['outlet', 'table'])
                ->get();
        } else {
            // For guests
            $guestReservationIds = Session::get('guest_reservations', []);
            $reservations = Reservation::whereIn('id', $guestReservationIds)
                ->orderBy('reservation_date', 'desc')
                ->orderBy('reservation_time', 'desc')
                ->with(['outlet', 'table'])
                ->get();
        }

        return view('customer.reservations.index', compact('reservations'));
    }

    /**
     * Show specific reservation detail
     */
    public function show($reservationNumber)
    {
        $reservation = Reservation::where('reservation_number', $reservationNumber)
            ->with(['outlet', 'table'])
            ->firstOrFail();

        // Check if user has access to this reservation
        if (Auth::check()) {
            if ($reservation->user_id !== Auth::id()) {
                abort(403, 'Unauthorized access to reservation');
            }
        } else {
            // For guest, check session
            $guestReservationIds = Session::get('guest_reservations', []);
            if (!in_array($reservation->id, $guestReservationIds)) {
                abort(403, 'Unauthorized access to reservation');
            }
        }

        return view('customer.reservations.show', compact('reservation'));
    }

    /**
     * Cancel reservation
     */
    public function cancel($reservationNumber)
    {
        $reservation = Reservation::where('reservation_number', $reservationNumber)
            ->firstOrFail();

        // Check if user has access
        if (Auth::check()) {
            if ($reservation->user_id !== Auth::id()) {
                abort(403, 'Unauthorized');
            }
        } else {
            $guestReservationIds = Session::get('guest_reservations', []);
            if (!in_array($reservation->id, $guestReservationIds)) {
                abort(403, 'Unauthorized');
            }
        }

        // Check if can be cancelled
        if (!$reservation->canBeCancelled()) {
            return back()->with('error', 'Reservasi tidak dapat dibatalkan karena status: ' . $reservation->status);
        }

        // Check if not too late to cancel (e.g., at least 2 hours before)
        $reservationDateTime = Carbon::parse($reservation->reservation_date->format('Y-m-d') . ' ' . $reservation->reservation_time->format('H:i:s'));
        if ($reservationDateTime->diffInHours(now()) < 2 && $reservationDateTime->isFuture()) {
            return back()->with('error', 'Reservasi tidak dapat dibatalkan kurang dari 2 jam sebelum waktu reservasi.');
        }

        $reservation->update([
            'status' => Reservation::STATUS_CANCELLED,
            'cancelled_at' => now(),
        ]);

        Log::info('Customer cancelled reservation', [
            'reservation_number' => $reservation->reservation_number,
            'cancelled_by' => Auth::check() ? Auth::user()->name : 'guest',
        ]);

        // TODO: Send cancellation email

        return redirect()->route('customer.reservations.index')
            ->with('success', 'Reservasi berhasil dibatalkan.');
    }

    /**
     * Generate time slots
     */
    private function generateTimeSlots($startTime, $endTime, $intervalMinutes)
    {
        $slots = [];
        $start = Carbon::createFromFormat('H:i', $startTime);
        $end = Carbon::createFromFormat('H:i', $endTime);

        while ($start <= $end) {
            $slots[] = $start->format('H:i');
            $start->addMinutes($intervalMinutes);
        }

        return $slots;
    }
}
