<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Category extends Model
{
    use HasFactory;

    // Category Type Constants
    const TYPE_FOOD = 'food';
    const TYPE_DRINK = 'drink';
    const TYPE_SNACK = 'snack';
    const TYPE_DESSERT = 'dessert';
    const TYPE_OTHER = 'other';

    protected $fillable = [
        'outlet_id',
        'name',
        'type',
        'description',
        'image',
        'status',
    ];

    /**
     * Get all available category types
     */
    public static function getTypes(): array
    {
        return [
            self::TYPE_FOOD => 'Makanan',
            self::TYPE_DRINK => 'Minuman',
            self::TYPE_SNACK => 'Snack',
            self::TYPE_DESSERT => 'Dessert',
            self::TYPE_OTHER => 'Lainnya',
        ];
    }

    /**
     * Get type label in Indonesian
     */
    public function getTypeLabelAttribute(): string
    {
        $types = self::getTypes();
        return $types[$this->type] ?? ucfirst($this->type);
    }

    /**
     * Scope a query to only include categories of a specific type
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * Get the outlet that owns the Category
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function outlet(): BelongsTo
    {
        return $this->belongsTo(Outlet::class);
    }

    /**
     * Get all of the products for the Category
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function products(): HasMany
    {
        return $this->hasMany(Product::class);
    }
}
