<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Reservation extends Model
{
    use HasFactory;

    // Reservation Status Constants
    const STATUS_PENDING = 'pending';
    const STATUS_CONFIRMED = 'confirmed';
    const STATUS_CANCELLED = 'cancelled';
    const STATUS_COMPLETED = 'completed';
    const STATUS_NO_SHOW = 'no_show';

    protected $fillable = [
        'reservation_number',
        'user_id',
        'guest_info',
        'outlet_id',
        'table_id',
        'reservation_date',
        'reservation_time',
        'party_size',
        'status',
        'special_requests',
        'notes',
        'confirmed_at',
        'cancelled_at',
    ];

    protected $casts = [
        'guest_info' => 'array',
        'reservation_date' => 'date',
        'reservation_time' => 'datetime:H:i',
        'confirmed_at' => 'datetime',
        'cancelled_at' => 'datetime',
        'party_size' => 'integer',
    ];

    /**
     * Get all available reservation statuses
     */
    public static function getStatuses(): array
    {
        return [
            self::STATUS_PENDING => 'Menunggu Konfirmasi',
            self::STATUS_CONFIRMED => 'Dikonfirmasi',
            self::STATUS_CANCELLED => 'Dibatalkan',
            self::STATUS_COMPLETED => 'Selesai',
            self::STATUS_NO_SHOW => 'Tidak Hadir',
        ];
    }

    /**
     * Check if reservation can be confirmed
     */
    public function canBeConfirmed(): bool
    {
        return $this->status === self::STATUS_PENDING;
    }

    /**
     * Check if reservation can be cancelled
     */
    public function canBeCancelled(): bool
    {
        return in_array($this->status, [self::STATUS_PENDING, self::STATUS_CONFIRMED]);
    }

    /**
     * Check if reservation is active (not cancelled or completed)
     */
    public function isActive(): bool
    {
        return in_array($this->status, [self::STATUS_PENDING, self::STATUS_CONFIRMED]);
    }

    /**
     * Get customer name (from user or guest_info)
     */
    public function getCustomerNameAttribute(): string
    {
        if ($this->user) {
            return $this->user->name;
        }
        return $this->guest_info['name'] ?? 'Guest';
    }

    /**
     * Get customer email (from user or guest_info)
     */
    public function getCustomerEmailAttribute(): ?string
    {
        if ($this->user) {
            return $this->user->email;
        }
        return $this->guest_info['email'] ?? null;
    }

    /**
     * Get customer phone (from user or guest_info)
     */
    public function getCustomerPhoneAttribute(): ?string
    {
        if ($this->user) {
            return $this->user->phone ?? null;
        }
        return $this->guest_info['phone'] ?? null;
    }

    /**
     * Get formatted date time
     */
    public function getFormattedDateTimeAttribute(): string
    {
        return $this->reservation_date->format('d M Y') . ' - ' . $this->reservation_time->format('H:i');
    }

    /**
     * Scope to get upcoming reservations
     */
    public function scopeUpcoming($query)
    {
        return $query->where('reservation_date', '>=', now()->toDateString())
                     ->whereIn('status', [self::STATUS_PENDING, self::STATUS_CONFIRMED])
                     ->orderBy('reservation_date')
                     ->orderBy('reservation_time');
    }

    /**
     * Scope to get today's reservations
     */
    public function scopeToday($query)
    {
        return $query->whereDate('reservation_date', now()->toDateString());
    }

    /**
     * Get the user that owns the Reservation
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the outlet that owns the Reservation
     */
    public function outlet(): BelongsTo
    {
        return $this->belongsTo(Outlet::class);
    }

    /**
     * Get the table that owns the Reservation
     */
    public function table(): BelongsTo
    {
        return $this->belongsTo(Table::class);
    }
}
