<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     * 
     * Creates reservations table for table booking feature
     */
    public function up(): void
    {
        Schema::create('reservations', function (Blueprint $table) {
            $table->id();
            $table->string('reservation_number')->unique()->comment('Unique reservation code');
            
            // Customer info (nullable user_id for guest reservations)
            $table->foreignId('user_id')->nullable()->constrained()->onDelete('set null');
            $table->text('guest_info')->nullable()->comment('JSON: Guest name, email, phone if not registered user');
            
            // Reservation details
            $table->foreignId('outlet_id')->constrained()->onDelete('cascade');
            $table->foreignId('table_id')->nullable()->constrained()->onDelete('set null')->comment('Null if not yet assigned');
            
            $table->date('reservation_date')->comment('Date of reservation');
            $table->time('reservation_time')->comment('Time slot for reservation');
            $table->integer('party_size')->default(2)->comment('Number of people');
            
            // Status management
            $table->enum('status', ['pending', 'confirmed', 'cancelled', 'completed', 'no_show'])
                  ->default('pending')
                  ->comment('pending: awaiting confirmation, confirmed: approved by cashier, completed: customer arrived and finished, cancelled: cancelled by customer or cashier, no_show: customer did not show up');
            
            // Additional info
            $table->text('special_requests')->nullable()->comment('Special requests from customer');
            $table->text('notes')->nullable()->comment('Internal notes from cashier');
            
            // Timestamps for status changes
            $table->timestamp('confirmed_at')->nullable();
            $table->timestamp('cancelled_at')->nullable();
            $table->timestamps();
            
            // Indexes for performance
            $table->index(['reservation_date', 'reservation_time'], 'idx_date_time');
            $table->index('status', 'idx_status');
            $table->index('outlet_id', 'idx_outlet');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('reservations');
    }
};
